
// ----- Logistic Orbit Diagram (double) - Processing -----
// include shifts - 1/4 screen in each direction?
// include simple zoom in and out - factor of 2?
// include option to see 0 to 4

// ----- Define the interface design layout -----

PFont font1,font2,font3;

// --- Palette - set colors ---

public color backClr= color(17, 51, 34) ;

public color Black  = color(  0) ;
public color drkGray= color( 51) ;
public color drkMidG= color(102) ;
public color midGray= color(127) ;
public color litMidG= color(153) ;
public color litGray= color(204) ;
public color White  = color(255) ;

public color Red    = color(255,  0,  0) ;
public color Yellow = color(255,255,  0) ;
public color Green  = color(  0,255,  0) ;
public color Cyan   = color(  0,255,255) ;
public color Blue   = color(  0,  0,255) ;
public color Magenta= color(255,  0,255) ;


// --- define functions and platting vars ---

public double f(double x, double r) {
  return r*x*(1-x);
}

public int wx,wy,oldwx,oldwy;
public double fx,fy;
public color fncClr = Yellow;

// ----- application window data and method -----

String title= "The Iterated Logistic Orbit Diagram";
int
winWid  = 900, winHgt = 650,
workLft =   5, workRgt= winWid-5,
workTop =  25, workBas= winHgt-5,
workMidX= round((workLft+workRgt)/2),
mx, my;

void windowInit() {
  stroke(litMidG);
  line(workLft,workTop-1, workRgt,workTop-1);
  stroke(Black);
  line(workLft,workTop-2, workRgt,workTop-2);
  textFont(font2);
  plotTextCJ(workMidX,workTop-7, title, litGray);
}

// ----- graph plane window data and method -----

Plane w1;  // instance name for window 1

// --- plane pixel window integer bounds ---
int
w1wTop= 50+workTop, w1wLft= 150+workLft,
w1wWid= 600, w1wHgt= 400,
w1wRgt= w1wLft + w1wWid, w1wBas= w1wTop + w1wHgt;

// --- function float bounds ---
double
w1fLft= 2.8, w1fRgt= 4.000001, w1fBas= 0, w1fTop= 1;

// --- axis tiks and labels ---
float
w1xTikS= 0.1, w1xTikL= 0.2, w1xLabels= 0.2,
w1yTikS= 0.1, w1yTikL= 0.5, w1yLabels= 0.5;
String
w1xName= "r", w1yName= "x", txt= "";

// --- axis value precision ---
int
w1xPlaces= 3, w1yPlaces= 3, zoomPlaces = 14;

void w1Init() {
  w1 = new Plane(w1wLft, w1wRgt, w1fLft, w1fRgt, w1xTikS, w1xTikL, w1xLabels, w1xPlaces, w1xName, 
                 w1wBas, w1wTop, w1fBas, w1fTop, w1yTikS, w1yTikL, w1yLabels, w1yPlaces, w1yName);
  w1.setPlaneVars();
  w1.drawPlane();
  textFont(font2);
  plotTextLJ(w1wRgt+5,w1wBas+3,"r",litGray);
  plotTextCJ(w1wLft+1,w1wTop-5,"x",fncClr);
  txt= "xn+1 = rxn(1-xn)";
  int len= (int) textWidth(txt);
  txt= "x_[n+1] = rx_[n](1-x_[n])";
  plotTextSupSubLJ(w1.wMidx-len/2,workTop+30, txt, Yellow) ; 
  textFont(font2);
  drawGraph();
}  

void clearLabels() {
  boxClear(w1.wLft-10,w1.wRgt+10,w1.wBas+20,w1.wBas+1);
  boxClear(workLft,w1.wLft-1,w1.wBas+5,w1.wTop-5);
}

void defaultValues() {  // graph plane reset object defaults
  w1.fLft = 2.8;
  w1.fRgt = 4.00001;
  w1.fBas = 0;
  w1.fTop = 1;
  iterates= 100;
  x0 = 0.5;
}
//----- end of graph plane data and methods -----


// ----- h sliders -----
// --- h1 instance slider data and method ---
HSlider h1;  // instance name

int     // window pixel values
h1Axis  = w1wBas + 120,
h1wWid  = 500,
h1wLft  = w1wLft, h1wRgt  = h1wLft+h1wWid;

public float iterates= 100;  // default value
String h1Name= "iterates";
int h1Places= 0;

float   // float function values
h1fLft=   0, h1fRgt= 10000,
h1TikS= 100, h1TikL=   500, h1Labels= 1000, h1Steps= 100;  // round nearest over tik marks


void h1Init() {
  h1 = new HSlider(h1wLft,h1wRgt, h1Axis, h1fLft,h1fRgt, h1TikS, h1TikL, h1Labels, h1Name, h1Steps, h1Places, White);
  h1.DrawSlider(iterates);
}  // --- end h1 data and method ---


// ----- v sliders -----
// --- v1 instance slider data and method ---
VSlider v1;  // instance name

int     // window pixel values
v1Axis  = w1wRgt + 70,
v1wHgt  = 200,
v1wBas  = w1wBas, v1wTop  = w1wTop;

public float x0= 0.5;
String v1Name= "";
int v1Places= 3;

float   // float function values
v1fBas= 0, v1fTop= 1,
v1TikS= 0.1, v1TikL= 0.5, v1Labels= 0.5, v1Steps= 0.1;  // round nearest over tik marks


void v1Init() {
  v1 = new VSlider(v1wBas,v1wTop, v1Axis, v1fBas,v1fTop, v1TikS, v1TikL, v1Labels, v1Name, v1Steps, v1Places, White);
  v1.DrawSlider(x0);
  plotTextSupSubLJ(v1.Axis,v1.wBas+15,"x_[0]",White) ;
}  // --- end v1 data and method ---


// ----- drag rectangle data -----
int x1,y1, x2,y2;                   // drag rect corners
int mLft,mRgt,mBas,mTop;            // drag rect pixel bounds
double yTop, yBas, xLft, xRgt;      // temp drag rect function bounds
boolean zoom = false;


// --- rollover textbox
int xyLft   = w1wLft;               // rollover x,y text box
int xyTop   = w1wBas + 27;
int xyBas   = xyTop + 40;
int xyBasLn1= xyTop + 16;
int xyBasLn2= xyBasLn1 + 20;
int len1,len2,xyPlaces;
int xyTxtEqx,xyTxtRgt,xyRgt;
color xyClr = White;
boolean clearFlag;

void xyInit() {    // rollover values
  textFont(font2);
  len1    = (int) textWidth("y = ");     // max label width
  len2    = (int) textWidth("-0.00000000000000");    // max number width
  xyPlaces= 14;
  xyTxtEqx= xyLft + 2 + len1;
  xyTxtRgt= xyTxtEqx + len2;
  xyRgt   = xyTxtRgt + 5;
}

void xyRollOver() {   // rollover location lines
  stroke(White);      // rollover color
  line(mouseX,w1wTop+1, mouseX,w1wBas-1);  // vertical
  line(w1wLft+1,mouseY, w1wRgt-1,mouseY);  // horizontal
  xyText() ;  
}

void xyText() {
  xyClear();
  textFont(font2);
  plotTextRJ( xyTxtEqx,xyBasLn1,"x = ",xyClr) ;
  fx = (double) w1.xPixelToReal(mouseX);
  if (!zoom) {     // fix numerics
    fx = roundPlaces(fx,3);
  }
  txt= doubleToStringSigFigures(fx,xyPlaces);
  plotTextLJ( xyTxtEqx,xyBasLn1,txt,xyClr) ;

  plotTextRJ( xyTxtEqx,xyBasLn2,"y = ",xyClr) ;
  fy = (double) w1.yPixelToReal(mouseY);
  if (!zoom) {    // fix numerics
    fy = roundPlaces(fy,3);
  }
  txt= doubleToStringSigFigures(fy,xyPlaces);
  plotTextLJ( xyTxtEqx,xyBasLn2,txt,xyClr) ;
}

void xyTextClear() {
  boxClear(xyTxtEqx+2,xyRgt,xyBas,xyTop);
}
void xyClear() {
  boxClear(xyLft,xyRgt,xyBas,xyTop);
}

// ----- buttons -----

// --- reset button ---
Button rs;    // button class rs instance
int
rsWid= 55,
rsLft= w1wRgt - rsWid,
rsTop= w1wBas + 30;          
String rsLabel= "Reset";
void rsInit() {
  rs = new Button(rsLft, rsTop, rsWid, rsLabel);
  rs.drawButton();
}

// --- clear transients button ---
Button ct;    // button class ct instance
int
ctWid= 130,
ctLft= rsLft - ctWid - 2,
ctTop= rsTop;         
String ctLabel= "Clear Transients";
void ctInit() {
  ct = new Button(ctLft, ctTop, ctWid, ctLabel);
  ct.drawButton();
}

// --- total iterations text ---
int itrTxtLft, itrTxtBas;
color itrClr;
int iterations= 0;
void iterationsText() {
  itrClr   = Yellow;
  itrTxtLft= ctLft;
  itrTxtBas= ctTop+45;
  textFont(font2);
  txt= "Iterations: " + str(iterations);
  boxClear(itrTxtLft,workRgt,itrTxtBas+2,itrTxtBas-14);
  plotTextLJ( itrTxtLft,itrTxtBas,txt,itrClr);
}

// ----- End of Design Layout Data and Methods -----

//--- set default values and intialize graphics ---

double[] xold = new double[w1wWid+1];    // last iterate x for each r
int xoldPntr(int wx) {
  return wx-w1wLft;
}
PImage img  = createImage(w1wWid+1,w1wHgt+1,RGB); 


// ----- Main Program -----

void setup() {
  size(900, 650);
//  size(winWid,winHgt);
  frameRate(30);
  noSmooth();
  background(backClr);
  font1= loadFont("Tahoma-12.vlw");
  font2= loadFont("Tahoma-Bold-14.vlw");
  font3= loadFont("Tahoma-Bold-10.vlw");
  windowInit();
  w1Init();
  h1Init();
  v1Init();
  rsInit();
  ctInit();
  xyInit();
}

// ----- Main Event Loop -----

void draw() {
}

//--- mouse event handlers ---

void mouseMoved () {
  if (w1.planeWithin(mouseX,mouseY)) {       // rollover
    image(img, w1wLft,w1wTop);  // refresh plane image to clear
    xyRollOver();
    clearFlag = true;
  } // end if
  else if (clearFlag) {
    image(img, w1wLft,w1wTop);  // refresh rect (img) to clear
    xyClear();
    clearFlag = false;
  } // end else if
}  // ----- end of mouseMoved -----

void mousePressed () {
  if (w1.planeWithin(mouseX,mouseY)) {    // drag rect
    x1 = mouseX;
    y1 = mouseY;
  }
  else if (h1.Within(mouseX,mouseY)) {
    h1.lockSlider();
    iterates = h1.wSetValue(mouseX);
  }
  else if (v1.Within(mouseX,mouseY)) {
    v1.lockSlider();
    x0 = v1.wSetValue(mouseY);
  }
  else if (rs.Within()) {       // reset button
    rs.buttonDown() ;
  }
  else if (ct.Within()) {  // clear button
    ct.buttonDown() ;
  }
} // ----- end mousePressed -----


void mouseDragged() {
  if (w1.planeWithin(mouseX,mouseY)) {    // drag rect
    xyText() ;

    x2  = mouseX;
    y2  = mouseY;
    mLft= min(x1,x2);
    mRgt= max(x1,x2);
    mTop= min(y1,y2);
    mBas= max(y1,y2);

    image(img, w1wLft,w1wTop);  // refresh rect (img) to clear
    boxLines(mLft,mRgt,mBas,mTop,White);
  }

  if (h1.locked) {
    iterates = h1.wSetValue(mouseX);
  }
  if (v1.locked) {
    x0 = v1.wSetValue(mouseY);
  }
}  // ----- end of mouseDragged -----

void mouseReleased() {
  if (w1.planeWithin(mouseX,mouseY)) {    // drag rect
    if (x1==mouseX) { // no h move - width 10 pixels
      mLft= x1-5;
      mRgt= x1+5;
    }
    if (y1==mouseY) { // no v move - height 10 pixels
      mTop= y1-5;
      mBas= y1+5;
    }

    xLft = w1.xPixelToReal(mLft);  // temp real drag bounds
    xRgt = w1.xPixelToReal(mRgt); 
    yTop = w1.yPixelToReal(mTop);
    yBas = w1.yPixelToReal(mBas);

    w1.fLft = xLft;  // reset plane function bounds to real drag bounds
    w1.fRgt = xRgt;
    w1.fBas = yBas;
    w1.fTop = yTop;
    w1.setPlaneVars();

    zoom= true;
    zoomPlane();
    drawGraph();
  }
  else if (rs.Within()) {       // reset button
    rs.buttonUp() ;
    zoom = false;
    defaultValues();
    /*
    w1.fLft = 2.8;
    w1.fRgt = 4.000001;
    w1.fBas = 0;
    w1.fTop = 1;
    iterates= 100;
    x0= 0.5;
*/
    h1.fSetValue(iterates);
    v1.fSetValue(x0);

    w1.setPlaneVars();
    clearLabels();
    w1.drawPlane();
    drawGraph();
  }
  else if (ct.Within()) {  // clear transients button
    ct.buttonUp() ;
    clearTransients();
  }

  else if (h1.locked) {    // h1 slider
    drawGraph();
    h1.unlockSlider();
  }
  else if (v1.locked) {    // v1 slider
    drawGraph();
    v1.unlockSlider();
  }
}  // ----- end of mouseReleased -----


// ----- graph plane methods -----

void drawGraph() {
  w1.clearPlane() ;
  stroke(Yellow);
  double r;
  for (int wx = w1wLft+1; wx<w1wRgt; wx ++) {  // r loop
    fx= x0 ;
    r = w1.xPixelToReal(wx);

    for (int i=0; i<iterates; i++) {  // iterate
      fx = f(fx,r);
      wy = w1.yRealToPixel(fx);
      if (w1.planeWithin(wx,wy)) {    // point on plane
        set(wx,wy,Yellow);
      } // next i
    }
    // xoldPntr = wx-w1wLft;
    xold[xoldPntr(wx)] = fx;
  } // next wx
  iterations= (int) iterates;
  iterationsText();
  img = get(w1wLft,w1wTop, w1wWid+1,w1wHgt+1);  // save image for rollover
}

void clearTransients() {
  w1.clearPlane() ;
  stroke(Yellow) ;
  double r ;

  for (int wx = w1wLft+1; wx<w1wRgt; wx ++) {  // r loop
    r   = w1.xPixelToReal(wx);
    // xoldPntr= wx-w1wLft;
    fx  = xold[xoldPntr(wx)];

    for (int i=0; i<1000; i++) {      // clear transients
      fx = f(fx,r);
    }

    for (int i=0; i<iterates; i++) {  // draw iterates
      fx = f(fx,r);
      wy = w1.yRealToPixel(fx);
      if (w1.planeWithin(wx,wy)) {
        set(wx,wy,Yellow);
      }
    }
    // xoldPntr = wx-w1wLft;
    xold[xoldPntr(wx)] = fx;
  } // next wx
  iterations= iterations + (int) iterates;
  iterationsText();
  img = get(w1wLft,w1wTop, w1wWid,w1wHgt);  // save image for rollover
}

void zoomPlane() {
  w1.clearPlane() ;
  clearLabels() ;
  // typeset the boundary values
  textFont(font1);
  txt = doubleToStringSigFigures(w1.fLft, zoomPlaces);      // fLft
  plotTextLJ( w1.wLft,w1.wBas+18,txt,litGray);
  txt = doubleToStringSigFigures(w1.fRgt, zoomPlaces);      // fRgt
  plotTextRJ( w1.wRgt,w1.wBas+18,txt,litGray);

  txt = doubleToStringSigFigures(w1.fBas, zoomPlaces);      // fBas
  plotTextRJ(w1.wLft-4,w1.wBas,txt,litGray);
  txt = doubleToStringSigFigures(w1.fTop, zoomPlaces);      // fTop
  plotTextRJ(w1.wLft-4,w1.wTop+12,txt,litGray);
}
// ---- End of Tool Specific Code -----

// --- Public Library of Utilities?? ---

// --- round to nearest multiple of n ---
// --- double precision round ---
public double roundPlaces(double n, int places) {
  long power = (long) Math.pow(10,places);
  n = n*power;
  long rndn = Math.round(n);
  n = (double) rndn/power;
  return (double) n;
} // end roundPlaces

public double roundPlaces(float n, int places) {
  int power = (int) Math.pow(10,places);
  n = n*power;
  int rndn = Math.round(n);
  n = (float) rndn/power;
  return (float) n;
} // end roundPlaces

// --- round to nearest multiple of n ---
public float roundn(float n, int nearest) {
  n = n/nearest;
  int rn = round(n);  //Math.round()??
  n = (float) rn*nearest;
  return n;
} // end roundn

public float roundn(float n, float nearest) {
  n = n/nearest;
  int rn = round(n);
  n = (float) rn*nearest;
  return n;
} // end roundn


// --- typesetting routines ---


public int findChar(String s, char ch) {  // char location in a string
  int l = s.length();
  int chpos = -1;
  char c;
  for (int i=0; i<l; i ++) {
    c = s.charAt(i);
    if (c==ch) {
      chpos = i;
      i = l;      // exit for
    } // end if
  }   // end for
  return chpos;
}

void plotTextSupSubLJ(int txtx, int txtBas, String txt, int txtClr) {
  // superscript typesetting - finds _[] and ^[]
  int last = txt.length();
  int pos0 = findChar(txt,'^');
  int pos1 = findChar(txt,'_');
  int pos  = 0;
  int p;
  // setTextFont(1,14,"bold");
  textFont(font2);
  if ((pos0==-1) && (pos1==-1)) {         // no super or sub
    plotTextLJ(txtx,txtBas,txt,txtClr); 
  }  // plot and exit
  else {
    if (pos0==-1) {          // sub
      pos = pos1; 
    }
    else if (pos1==-1) {     // super
      pos = pos0;  
    }
    else {
      pos = min(pos0,pos1);  // both - use nearest
    }

    p = findChar(txt,']');                   // end of subsuper chars
    if (p>-1) {
      String slft = txt.substring(0,pos);    // get left string
      int lftlen  = (int) textWidth(slft);   // left length in pixels
      plotTextLJ(txtx,txtBas,slft,txtClr);

      String sbsp = txt.substring(pos+2,p);  // get subsuper chars
      textFont(font3);
      // setTextFont(1,12,"bold");
      if (pos0==pos) {
        plotTextLJ(txtx+lftlen,txtBas-6,sbsp,txtClr); 
      }
      else {
        plotTextLJ(txtx+lftlen,txtBas+5,sbsp,txtClr);
      }
      int sbsplen = (int) textWidth(sbsp);   // subsuper length in pixels
      textFont(font2);
      //setTextFont(1,14,"bold");

      String srgt = txt.substring(p+1,last); // get right string
      if (srgt.length()!=0) {                // any chars in right string?
        plotTextSupSubLJ (txtx+lftlen+sbsplen,txtBas,srgt,txtClr);
      }
    }
  } // end else
  textFont(font2);
}

public String doubleToStringSigFigures(double d, int sigfigs)
{
  d = roundPlaces(d,sigfigs);
  String s1 = Double.toString(d);
  int len   = s1.length();
  int loc   = s1.indexOf("E");
  int posdot= s1.indexOf(".");
  int pos;
  if (sigfigs < 1) return s1;

  if (loc != -1)	// There is an E in s1
  {
    pos = Math.min(posdot+sigfigs, loc);	
    return s1.substring(0,pos) +"E" + s1.substring(loc+1,len);
  }
  else	// no E in the string representation s1
  {
    pos = sigfigs + 1;
    if (s1.indexOf("-") == 0) pos += 1;
    if (posdot <= pos)	//  Need to keep figures right of decimal
    {	
      int dotMove = (Double.toString(d*1000.0).indexOf(".") - posdot);
      int leadZeroes = 3 - dotMove;	// count leading zeroes and adjust
      if (leadZeroes > 0) pos += leadZeroes;
      pos = Math.min(pos, len);
      return s1.substring(0, pos);
    }
    int numZeroes = posdot - pos;
    String s2 = s1.substring(0, pos);
    for (int i = 0; i< numZeroes; i++) s2 = s2 + "0";
    return s2 + ".0";
  }
}  //  end doubleToStringSigFigures


// --- box drawing routines ---
// left, right, base, top

public void boxClear(int l, int r, int b, int t) {
  stroke(backClr);
  fill(backClr);
  rect(l,t,r-l,b-t);
}
public void boxArea(int l, int r, int b, int t, color clr) {
  stroke(clr);
  fill(clr);
  rect(l,t,r-l,b-t);
}
public void boxLines(int l, int r, int b, int t, color clr) {
  stroke(clr);
  noFill();
  rect(l,t,r-l,b-t);
}
public void boxRect(int l, int r, int b, int t, color areaClr, color edgeClr) {
  stroke(edgeClr);
  fill(areaClr);
  rect(l,t,r-l,b-t);
}

// --- type formatting routines ---

public void plotTextLJ(int lft,int txtBas,String txt,int txtClr) {
  textAlign(LEFT);
  fill(txtClr);
  text(txt,lft,txtBas);
}
public void plotTextRJ(int rgt,int txtBas,String txt,int txtClr) {
  textAlign(RIGHT);
  fill(txtClr);
  text(txt,rgt,txtBas);
}
public void plotTextCJ(int mid,int txtBas,String txt,int txtClr) {
  textAlign(CENTER);
  fill(txtClr);
  text(txt,mid,txtBas);
}
// ----- end of library routines -----


// ----- Graph Plane Class data and methods -----

class Plane {
  // PFont font1,font2;

  int wLft,wRgt,wBas,wTop;           // The window pixel values
  int wWid,wHgt;                     // Pixel width of slider
  double fLft, fRgt, fBas, fTop;     // Min and max float values
  double fWid,fHgt;                  // float width
  float xSTik, xLTik, xLabels; // tik marks and labels on x axis
  float ySTik, yLTik, yLabels; // tik marks and labels on y axis
  int xPlaces,yPlaces;
  int wX0, wY0;
  int wMidx, wMidy;
  String xName, yName;

  // --- constructor ---
  Plane(int wL, int wR, double fL, double fR, float xSTk, float xLTk, float xLab, int xplaces, String xname,
        int wB, int wT, double fB, double fT, float ySTk, float yLTk, float yLab, int yplaces, String yname) {
    wLft   = wL; wRgt   = wR;
    wBas   = wB; wTop   = wT;
    fLft   = fL; fRgt   = fR;
    fBas   = fB; fTop   = fT;
    xSTik  = xSTk; xLTik  = xLTk; xLabels= xLab;
    xName  = xname;
    ySTik  = ySTk; yLTik  = yLTk; yLabels= yLab;
    yName  = yname;
    xPlaces= xplaces; yPlaces= yplaces;
  }

  void setPlaneVars() {
    wWid = wRgt-wLft;
    wHgt = wBas-wTop;
    fWid = fRgt-fLft;
    fHgt = fTop-fBas;
    wY0  = yRealToPixel(0);
    wX0  = xRealToPixel(0);
    wMidx= (int) ((wLft+wRgt)/2);
    wMidy= (int) ((wTop+wBas)/2);
  }

  boolean planeWithin(int mx,int my) {
    return (mx>=wLft && mx<=wRgt && my<=wBas && my>=wTop) ;
  }

  void drawPlane() {
    setPlaneVars();
    clearPlane();
    textFont(font1);
    stroke(litGray);
    xTiksS();
    xTiksL();
    xLabels();
    yTiksS();
    yTiksL();
    yLabels();
  }

  void clearPlane() {
    boxRect(wLft,wRgt,wBas,wTop, Black,litGray) ;
    zeroAxes();
  }

  void zeroAxes() {
    stroke(midGray);         // line color blue
    if(0>fBas && 0<fTop) {
      line(wLft+1,wY0, wRgt-1,wY0);  // x-axis
    }
    if(0>fLft && 0<fRgt) {
      line(wX0,wTop+1, wX0,wBas-1);  // y-axis
    }
  }


  public double xPixelToReal(int xPixel) {
    return (fLft + fWid*(xPixel-wLft)/wWid);
  }  //  xPixelToReal

  public double yPixelToReal(int yPixel) {
    return (fBas + fHgt*(wBas-yPixel)/wHgt);
  }  //  yPixelToReal

  public int xRealToPixel(double xReal) {
    return (int)(wLft + wWid*(xReal-fLft)/fWid);
  }  //  xRealToPixel

  public int yRealToPixel(double yReal) {
    return (int)(wBas - wHgt*(yReal-fBas)/fHgt);
  }  //  yRealToPixel

  // --- plane axis drawing methods ---

  void xTiksS() {
    int xPix;
    for (double x=fLft; x<=fRgt; x+=xSTik) {  // short tik marks
      x = roundPlaces(x,xPlaces) ;
      xPix = (int)round((float)(wLft + wWid*(x-fLft)/fWid));
      line(xPix,wBas, xPix,wBas+2) ;
    }
  }
  void xTiksL() {
    int xPix;
    for (double x=fLft; x<=fRgt; x+=xLTik) {  // long tik marks
      x = roundPlaces(x,xPlaces) ;
      xPix = (int)round((float)(wLft + wWid*(x-fLft)/fWid));
      // println(xPix);
      line(xPix,wBas, xPix,wBas+4) ;
    }
  }
  void xLabels() { 
    int xPix;
    textFont(font1);
    for (double x=fLft; x<=fRgt; x+=xLabels) {   // labels tik marks
      x = (float) roundPlaces(x,xPlaces) ;
      xPix = (int)round((float)(wLft + wWid*(x-fLft)/fWid));
      if (x>=0) {
        xPix = xPix+1 ;
      }
      else {
        xPix = xPix-1 ;
      }
      plotTextCJ(xPix,wBas+18,str((float)x),litGray) ;   // int or float flag??
    }
  }

  // ----- axis drawing methods -----

  void yTiksS() {
    int yPix;
    for (double y=fBas; y<=fTop; y+=ySTik) {  // short tik marks
      y = (double) roundPlaces(y,yPlaces) ;
      yPix = (int)round((float)(wBas - wHgt*(y-fBas)/fHgt));
      line(wLft,yPix, wLft-2,yPix) ;
    }
  }
  void yTiksL() {
    int yPix;
    for (double y=fBas; y<=fTop; y+=yLTik) {  // short tik marks
      y = (double) roundPlaces(y,yPlaces) ;
      yPix = (int)round((float)(wBas - wHgt*(y-fBas)/fHgt));
      line(wLft,yPix, wLft-4,yPix) ;
    }
  }
  void yLabels() { 
    textFont(font1);
    int yPix;
    for (double y=fBas; y<=fTop; y+=yLabels) {  // labels tik marks
      y = (double) roundPlaces(y,yPlaces) ;
      yPix = (int)round((float)(wBas - wHgt*(y-fBas)/fHgt));
      plotTextRJ(wLft-6,yPix+4,str((float)y),litGray) ;   // int or float flag??
    }
  }
}
// --- end of Graph Plane Class methods ---


// ----- The Horizontal Slider Class -----

class HSlider {
  int wLft,wRgt,wBas,wTop,Axis;// The window pixel values
  int wWid;                    // Pixel width of slider
  int wx,oldwx;                // Pixel position of handle
  float fx;                    // float value of handle position
  float fLft, fRgt;            // Min and max float values
  float fWid;                  // float width
  float STiks, LTiks, Labels;  // tik marks and labels on axis
  String Name;                 // variable name
  float Step;                  // rounds to nearest over tik marks
  int Places;                  // decimal precision for rounding
  color varClr;                // color of Labels and value
  boolean locked;              // True when slider is active
  String txt;

  // --- constructor ---
  HSlider(int wL, int wR, int axY, float fL, float fR, float STk, float LTk, float label, String name, float stp, int plc, color clr) {
    Axis = axY;
    wLft = wL; wRgt = wR;
    wBas = Axis+15; wTop = Axis-15;
    wWid = wR-wL;

    fLft  = fL; fRgt  = fR;
    fWid  = fRgt-fLft;
    STiks = STk; LTiks = LTk;
    Labels= label;

    Name  = name;
    Step  = stp;
    Places= plc;
    varClr= clr;
  }
  
    // Draw the horizontal slider
  void DrawSlider(float fValue) { 
    stroke(litGray);
    line(wLft,Axis, wRgt,Axis);
    TiksS ();
    TiksL ();
    Labels();

    textFont(font2);
    plotTextRJ(wLft-8,wBas-1,Name,varClr);

    wx = RealToPixel(fValue);
    wSetValue(wx);
  }

  float wSetValue(int xPix) {
    float fx;
    wx = constrain(xPix, wLft, wRgt);
    fx = PixelToReal();            // set value text
    if (mouseY<Axis) {
      fx = roundn(fx,Step);
      fx = constrain(fx,fLft,fRgt);
      //fx = min(fx,fRgt);
      //fx = max(fx,fLft);
    }
    if (wx!=oldwx) {
      oldwx = wx;
      wx = RealToPixel(fx);

      boxClear(wLft-5,wRgt+60,wBas+1,Axis+1);      // erase handle and value
      stroke(litMidG);                      // draw slider track
      line(wLft,Axis+10, wRgt,Axis+10);
      stroke(Black);
      line(wLft,Axis+9, wRgt,Axis+9);

      boxArea(wx-1,wx+1,wBas-1,Axis+5,litGray);  // draw handle
      stroke(White);
      line(wx,Axis+2, wx,wBas);

      textFont(font2);                     // typeset value - hfont2?
      plotTextLJ(wRgt+8, wBas-1, nfc(fx, Places), varClr);
    }
    return fx;
  }

  void fSetValue(float fx) {
    fx = constrain(fx, fLft, fRgt);
    wx = RealToPixel(fx);            // set value text
    wSetValue(wx);
  }


  // Lock the interaction to the slider so the mouse can move off and still update
  void lockSlider() {
    locked = true;
  }

  // Release the slider connection
  void unlockSlider() {
    locked = false;
  }

  boolean Within(int mx, int my) {
    return ((mx >= wLft-10) && (mx <= wRgt+10) && (my > wTop) && (my < wBas)) ; // drag rect
  }

  // Returns the real value
  float PixelToReal() {
    return (fLft + fWid*(wx-wLft)/wWid);
  }  // end xPixelToReal

  // Returns the pixel value
  int RealToPixel(float xReal) {
    return round(wLft + wWid*(xReal-fLft)/fWid);
  }  // end xRealToPixel

  // --- slider axis drawing methods ---

  void TiksS() {
    for (float x=fLft; x<=fRgt; x+=STiks) {  // short tik marks
      x = (float) roundPlaces(x,Places) ;
      wx = RealToPixel(x) ;
      line(wx,Axis, wx,Axis-2) ;
    }
  }
  void TiksL() {
    for (float x=fLft; x<=fRgt; x+=LTiks) {  // long tik marks
      x = (float) roundPlaces(x,Places) ;
      wx = RealToPixel(x) ;
      line(wx,Axis, wx,Axis-4) ;
    }
  }
  void Labels() {
    textFont(font1);
    for (float x=fLft; x<=fRgt; x+=Labels) {  // labels tik marks
      if (x>=0) {   // positive
        wx = RealToPixel(x)+1 ;
      }
      else {        // negative
        wx = RealToPixel(x)-1 ;
      }
      if (Places>0) {
        txt = str((float)roundPlaces(x,Places));
      }
      else {
        txt = str((int)roundPlaces(x,Places));
      }
      plotTextCJ(wx,Axis-6,txt,litGray) ; 
    }
  }
} // end of hSlider Class



// ----- The Vertical Slider Class -----

class VSlider {
  int wBas,wTop,wLft,wRgt,Axis;// The window pixel values
  int wHgt;                    // Pixel width of slider
  int wy,oldwy;                // Pixel position of handle
  float fy;                    // float value of handle position
  float fBas, fTop;            // Min and max float values
  float fHgt;                  // float width
  float STiks, LTiks, Labels;  // tik marks and labels on axis
  String Name;                 // variable name
  float Step;                  // rounds to nearest over tik marks
  int Places;                  // decimal precision for rounding
  color varClr;                // color of Labels and value
  boolean locked;              // True when slider is active
  String txt;

  // --- constructor ---
  VSlider(int wB, int wT, int axX, float fB, float fT, float STk, float LTk, float label, String name, float stp, int plc, color clr) {
    Axis = axX;
    wBas = wB;
    wTop = wT;
    wRgt = Axis+15;
    wLft = Axis-15;
    wHgt = wB-wT;

    fBas  = fB;
    fTop  = fT;
    fHgt  = fTop-fBas;
    STiks = STk;
    LTiks = LTk;
    Labels= label;

    Name  = name;
    Step  = stp;
    Places= plc;
    varClr= clr;
  }
  
    // Draw the vertical slider
  void DrawSlider(float fValue) { 
    stroke(litGray);
    line(Axis,wBas, Axis,wTop);
    TiksS ();
    TiksL ();
    Labels();

    textFont(font2);
    plotTextCJ(Axis,wBas+20,Name,varClr);

    wy = RealToPixel(fValue);
    wSetValue(wy);
  }

  float wSetValue(int yPix) {
    wy = constrain(yPix, wTop, wBas);
    fy = PixelToReal(wy);            // set value text
    if (mouseX<Axis) {
      fy = roundn(fy,Step);
      fy = constrain(fy, fBas, fTop);
   }
    if (wy!=oldwy) {
      oldwy = wy;
      wy = RealToPixel(fy);

      boxClear(Axis+2,Axis+15,wBas+2,wTop-2);     // erase handle
      boxClear(Axis-30,Axis+30,wTop-6,wTop-22);   // value

      stroke(litMidG);                      // draw slider track
      line(Axis+10,wBas, Axis+10,wTop);
      stroke(Black);
      line(Axis+9,wBas, Axis+9,wTop);

      boxArea(Axis+5,wRgt-1,wy+1,wy-1,White);  // draw handle
      stroke(White);
      line(Axis+2,wy, wRgt,wy);

      textFont(font2);                     // typeset value
      plotTextCJ(Axis, wTop-10, nfc(fy, Places), varClr);
    }
    return fy;
  }

  void fSetValue(float fy) {
    fy = constrain(fy, fBas, fTop);
    wy = RealToPixel(fy);            // set value text
    wSetValue(wy);
  }


  // Lock the interaction to the slider so the mouse can move off and still update
  void lockSlider() {
    locked = true;
  }

  // Release the slider connection
  void unlockSlider() {
    locked = false;
  }

  boolean Within(int mx, int my) {
    return ((mx >= wLft-10) && (mx <= wRgt+10) && (my > wTop) && (my < wBas)) ; // drag rect
  }

  // Returns the real value
  float PixelToReal(int wy) {
    return (fBas + fHgt*(wBas-wy)/wHgt);
  }  // end yPixelToReal

  // Returns the pixel value
  int RealToPixel(float fy) {
    return round(wBas - wHgt*(fy-fBas)/fHgt);
  }  // end xRealToPixel

  // --- slider axis drawing methods ---

  void TiksS() {
    for (float y=fBas; y<=fTop; y+=STiks) {  // short tik marks
      y = (float) roundPlaces(y,Places) ;
      wy = RealToPixel(y) ;
      line(Axis,wy, Axis-2,wy) ;
    }
  }
  void TiksL() {
    for (float y=fBas; y<=fTop; y+=LTiks) {  // short tik marks
      y = (float) roundPlaces(y,Places) ;
      wy = RealToPixel(y) ;
      line(Axis,wy, Axis-4,wy) ;
    }
  }
  void Labels() {
    textFont(font1);
    for (float y=fBas; y<=fTop; y+=LTiks) {  // short tik marks
      if (y>=0) {   // positive
        wy = RealToPixel(y)+1 ;
      }
      else {        // negative
        wy = RealToPixel(y)-1 ;
      }
      if (Places>0) {
        txt = str((float)roundPlaces(y,Places));
      }
      else {
        txt = str((int)roundPlaces(y,Places));
      }
      plotTextRJ(Axis-6,wy+4,txt,varClr) ;   // int or float flag??
    }
  }
} // end of VSlider Class



// ----- Button Class -----

class Button {
  int Lft,Rgt,Bas,Top;   // The window pixel values
  int Wid,Mid,txtBas;                
  int Hgt = 21;
  String Label;
  int mx,my;

  // --- constructor ---
  Button(int btnLft, int btnTop, int btnWid, String btnLabel) {
    Lft  = btnLft;
    Top  = btnTop;
    Wid  = btnWid;
    Label= btnLabel;

    Rgt  = Lft+Wid;
    Bas  = Top+Hgt;
    Mid  = (Lft+Rgt)/2 + 1;
    txtBas = Bas - 5;
  }

  boolean Within() {
    mx = mouseX;
    my = mouseY; 
    return (mx>=Lft && mx<=Rgt && my<=Bas && my>=Top) ; // reset button
  }

  // --- button drawing routines ---
  void drawButton() {
    boxRect(Lft,Rgt,Bas,Top, drkGray,Black);
    stroke(White);
    line(Rgt,Top, Lft,Top);
    line(Lft,Bas, Lft,Top);
    textFont(font2);
    plotTextCJ(Mid,txtBas,Label,White);
  }
  void buttonDown() {
    stroke(Black);
    line(Rgt,Top, Lft,Top);
    line(Lft,Bas, Lft,Top);
    stroke(White);
    line(Rgt,Top, Rgt,Bas);
    line(Lft,Bas, Rgt,Bas);
  }
  void buttonUp() {
    stroke(White);
    line(Rgt,Top, Lft,Top);
    line(Lft,Bas, Lft,Top);
    stroke(Black);
    line(Rgt,Top, Rgt,Bas);
    line(Lft,Bas, Rgt,Bas);
  }
} // ----- end of Button class -----